import React, { useState } from 'react';
import { Calculator } from 'lucide-react';

export function SavingsCalculator() {
  const [originalPrice, setOriginalPrice] = useState<string>('');
  const [discountPercent, setDiscountPercent] = useState<string>('');
  const [additionalCoupon, setAdditionalCoupon] = useState<string>('');

  const calculateSavings = () => {
    const original = parseFloat(originalPrice) || 0;
    const discount = parseFloat(discountPercent) || 0;
    const coupon = parseFloat(additionalCoupon) || 0;
    
    const firstDiscount = original * (1 - discount / 100);
    const finalPrice = firstDiscount * (1 - coupon / 100);
    const totalSavings = original - finalPrice;
    
    return {
      finalPrice: finalPrice.toFixed(2),
      savings: totalSavings.toFixed(2),
      percentSaved: ((totalSavings / original) * 100).toFixed(1)
    };
  };

  const results = originalPrice ? calculateSavings() : null;

  return (
    <div className="bg-white rounded-lg shadow-lg p-6 max-w-md mx-auto">
      <div className="flex items-center gap-2 mb-6">
        <Calculator className="w-6 h-6 text-blue-600" />
        <h2 className="text-2xl font-bold text-gray-800">Savings Calculator</h2>
      </div>
      
      <div className="space-y-4">
        <div>
          <label className="block text-sm font-medium text-gray-700">Original Price ($)</label>
          <input
            type="number"
            value={originalPrice}
            onChange={(e) => setOriginalPrice(e.target.value)}
            className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
            placeholder="100.00"
          />
        </div>

        <div>
          <label className="block text-sm font-medium text-gray-700">Initial Discount (%)</label>
          <input
            type="number"
            value={discountPercent}
            onChange={(e) => setDiscountPercent(e.target.value)}
            className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
            placeholder="20"
          />
        </div>

        <div>
          <label className="block text-sm font-medium text-gray-700">Additional Coupon (%)</label>
          <input
            type="number"
            value={additionalCoupon}
            onChange={(e) => setAdditionalCoupon(e.target.value)}
            className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
            placeholder="10"
          />
        </div>

        {results && (
          <div className="mt-6 p-4 bg-blue-50 rounded-lg">
            <h3 className="text-lg font-semibold text-blue-900 mb-2">Your Savings</h3>
            <div className="space-y-2">
              <p className="text-blue-800">Final Price: <span className="font-bold">${results.finalPrice}</span></p>
              <p className="text-green-600">Total Savings: <span className="font-bold">${results.savings}</span></p>
              <p className="text-blue-800">Percent Saved: <span className="font-bold">{results.percentSaved}%</span></p>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}